-- =====================================================
-- DOC IC FORUM - DATABASE SCHEMA
-- Department of Corrections IC Roleplay System
-- =====================================================

-- Drop existing tables if they exist
DROP TABLE IF EXISTS activity_logs;
DROP TABLE IF EXISTS ctd_missions;
DROP TABLE IF EXISTS strike_points;
DROP TABLE IF EXISTS parole_forms;
DROP TABLE IF EXISTS bolingbroke_activities;
DROP TABLE IF EXISTS escorting_reports;
DROP TABLE IF EXISTS posts;
DROP TABLE IF EXISTS threads;
DROP TABLE IF EXISTS forum_categories;
DROP TABLE IF EXISTS users;
DROP TABLE IF EXISTS divisions;
DROP TABLE IF EXISTS ranks;

-- =====================================================
-- RANKS TABLE - 13 Rank Hierarchy
-- =====================================================
CREATE TABLE ranks (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(50) NOT NULL,
    display_name VARCHAR(100) NOT NULL,
    level INT NOT NULL,
    permissions TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_level (level),
    INDEX idx_level (level)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- DIVISIONS TABLE - Department Divisions
-- =====================================================
CREATE TABLE divisions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    code VARCHAR(20) NOT NULL UNIQUE,
    description TEXT,
    parent_division_id INT DEFAULT NULL,
    supervisor_rank_min INT DEFAULT 7 COMMENT 'Minimum rank level to supervise (7=LT, 8=Captain, 9=Commaderi, 10=Commandera, 11=majora, 12=majorb)',
    is_default TINYINT(1) DEFAULT 0 COMMENT '1 for Operations (default division)',
    has_menu TINYINT(1) DEFAULT 0 COMMENT '1 if division has its own menu header',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (parent_division_id) REFERENCES divisions(id) ON DELETE SET NULL,
    INDEX idx_code (code),
    INDEX idx_default (is_default)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- USERS TABLE
-- =====================================================
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) UNIQUE NOT NULL,
    ic_name VARCHAR(100) NOT NULL,
    rank_id INT NOT NULL,
    division_id INT DEFAULT 1 COMMENT 'Default to Operations division',
    badge_number VARCHAR(20) DEFAULT '-',
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100) UNIQUE,
    status ENUM('active', 'inactive', 'suspended', 'pending') DEFAULT 'active',
    last_login DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (rank_id) REFERENCES ranks(id) ON DELETE RESTRICT,
    FOREIGN KEY (division_id) REFERENCES divisions(id) ON DELETE SET NULL,
    INDEX idx_username (username),
    INDEX idx_rank (rank_id),
    INDEX idx_division (division_id),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- FORUM CATEGORIES
-- =====================================================
CREATE TABLE forum_categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    description TEXT,
    icon VARCHAR(100),
    display_order INT DEFAULT 0,
    required_rank INT DEFAULT 1,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_order (display_order),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- FORUM THREADS
-- =====================================================
CREATE TABLE threads (
    id INT AUTO_INCREMENT PRIMARY KEY,
    category_id INT NOT NULL,
    user_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    is_pinned TINYINT(1) DEFAULT 0,
    is_locked TINYINT(1) DEFAULT 0,
    views INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES forum_categories(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_category (category_id),
    INDEX idx_user (user_id),
    INDEX idx_pinned (is_pinned),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- FORUM POSTS
-- =====================================================
CREATE TABLE posts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    thread_id INT NOT NULL,
    user_id INT NOT NULL,
    content TEXT NOT NULL,
    is_edited TINYINT(1) DEFAULT 0,
    edited_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (thread_id) REFERENCES threads(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_thread (thread_id),
    INDEX idx_user (user_id),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- ESCORTING REPORTS (formerly Escort Logs)
-- =====================================================
CREATE TABLE escorting_reports (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    ic_escorting VARCHAR(100) NOT NULL COMMENT 'IC name of person being escorted',
    officer_involved TEXT COMMENT 'Officers involved in escort',
    mdt_case_id VARCHAR(50) COMMENT 'MDT case reference',
    from_location VARCHAR(100),
    to_location VARCHAR(100),
    escort_time DATETIME,
    debrief TEXT COMMENT 'Detailed debrief',
    status ENUM('draft', 'submitted', 'approved', 'rejected') DEFAULT 'draft',
    reviewed_by INT,
    reviewed_at DATETIME,
    review_notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user (user_id),
    INDEX idx_status (status),
    INDEX idx_escort_time (escort_time)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- BOLINGBROKE ACTIVITIES (formerly Incident Reports)
-- =====================================================
CREATE TABLE bolingbroke_activities (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    shift_leader VARCHAR(100) NOT NULL,
    officer_involved TEXT COMMENT 'Officers involved',
    activity_datetime DATETIME NOT NULL,
    debrief TEXT NOT NULL COMMENT 'Activity debrief',
    status ENUM('draft', 'submitted', 'approved', 'rejected') DEFAULT 'draft',
    reviewed_by INT,
    reviewed_at DATETIME,
    review_notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user (user_id),
    INDEX idx_status (status),
    INDEX idx_activity_date (activity_datetime)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- PAROLE FORMS
-- =====================================================
CREATE TABLE parole_forms (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    inmate_name VARCHAR(100) NOT NULL,
    mdt_case_id VARCHAR(50),
    charge VARCHAR(255) NOT NULL,
    sentence_start DATE NOT NULL,
    parole_conditions TEXT NOT NULL,
    end_of_parole DATE NOT NULL,
    officer_leader VARCHAR(100) NOT NULL,
    status ENUM('draft', 'submitted', 'approved', 'rejected') DEFAULT 'draft',
    reviewed_by INT,
    reviewed_at DATETIME,
    review_notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user (user_id),
    INDEX idx_status (status),
    INDEX idx_end_of_parole (end_of_parole)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- ACTIVITY LOGS
-- =====================================================
CREATE TABLE activity_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    action VARCHAR(100),
    description TEXT,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user (user_id),
    INDEX idx_action (action),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- STRIKE POINTS (Internal Affairs Division)
-- =====================================================
CREATE TABLE strike_points (
    id INT AUTO_INCREMENT PRIMARY KEY,
    target_user_id INT NOT NULL COMMENT 'Officer receiving strike',
    issued_by INT NOT NULL COMMENT 'IA officer issuing strike',
    points INT NOT NULL DEFAULT 1 COMMENT 'Strike points (1-3)',
    reason TEXT NOT NULL,
    incident_date DATETIME NOT NULL,
    evidence_link VARCHAR(255) COMMENT 'Link to evidence (screenshots, videos)',
    status ENUM('active', 'appealed', 'dismissed', 'expired') DEFAULT 'active',
    expires_at DATE COMMENT 'Strike expiration date',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (target_user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (issued_by) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_target (target_user_id),
    INDEX idx_issued (issued_by),
    INDEX idx_status (status),
    INDEX idx_incident (incident_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- CTD MISSIONS (Correctional Tactical Division)
-- =====================================================
CREATE TABLE ctd_missions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    channel ENUM('SRU', 'BEA') NOT NULL COMMENT 'Special Response Unit or Bail Enforcer Agent',
    title VARCHAR(255) NOT NULL,
    description TEXT NOT NULL,
    assigned_by INT NOT NULL COMMENT 'Major Field Operation assigning mission',
    assigned_to VARCHAR(255) COMMENT 'CTD members assigned',
    priority ENUM('low', 'medium', 'high', 'critical') DEFAULT 'medium',
    status ENUM('pending', 'in_progress', 'completed', 'cancelled') DEFAULT 'pending',
    due_date DATETIME,
    completion_notes TEXT,
    completed_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (assigned_by) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_channel (channel),
    INDEX idx_status (status),
    INDEX idx_assigned_by (assigned_by),
    INDEX idx_priority (priority)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =====================================================
-- INSERT DEFAULT DATA
-- =====================================================

-- Insert Divisions (in correct order for foreign keys)
-- First: Root divisions (no parent)
INSERT INTO divisions (id, name, code, description, parent_division_id, supervisor_rank_min, is_default, has_menu) VALUES
(1, 'Operations', 'OPS', 'Default division - General operations and standard duties', NULL, 7, 1, 0),
(6, 'Administrations', 'ADMIN', 'Oversees PSB division', NULL, 10, 0, 0),
(7, 'Division Of Field Operation', 'DOFO', 'Oversees Operations and CTD', NULL, 10, 0, 0);

-- Second: Child divisions (with parent references)
INSERT INTO divisions (id, name, code, description, parent_division_id, supervisor_rank_min, is_default, has_menu) VALUES
(5, 'Professional Standard Bureau', 'PSB', 'Oversees HR and IA divisions', 6, 9, 0, 0),
(2, 'Human Resources', 'HR', 'Manages personnel, pending forms, and user status', 5, 8, 0, 1),
(3, 'Internal Affairs', 'IA', 'Investigates misconduct and manages strike points', 5, 8, 0, 1),
(4, 'Correctional Tactical Division', 'CTD', 'Special operations - SRU and BEA units', 7, 8, 0, 1);

-- Insert 15 Ranks (ID 1-15)
INSERT INTO ranks (id, name, display_name, level, permissions) VALUES
(1, 'Trainee', 'Trainee', 1, 'forum.view,forum.post,forms.escort,forms.bolingbroke'),
(2, 'Correctional Officer', 'Correctional Officer', 2, 'forum.view,forum.post,forms.escort,forms.bolingbroke'),
(3, 'Senior Correctional Officer', 'Senior Correctional Officer', 3, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole'),
(4, 'Corporal', 'Corporal', 4, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review'),
(5, 'Sergeant', 'Sergeant', 5, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review'),
(6, 'Sergeant Overwatch', 'Sergeant Overwatch', 6, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review'),
(7, 'Lieutenant', 'Lieutenant', 7, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review,admin.view,admin.users,admin.pending'),
(8, 'Captain', 'Captain', 8, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review,admin.view,admin.users,admin.pending'),
(9, 'Commaderi', 'Commander PSB', 9, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review,admin.view,admin.users,admin.pending'),
(10, 'Commandera', 'Commander COD', 10, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review,admin.view,admin.users,admin.pending'),
(11, 'majora', 'Major DOFO', 11, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review,admin.view,admin.users,admin.pending'),
(12, 'majorb', 'Major ADMIN', 12, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review,admin.view,admin.users,admin.pending'),
(13, 'Deputy Chief Warden', 'Deputy Chief Warden', 13, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review,admin.view,admin.users,admin.pending'),
(14, 'Warden', 'Warden', 14, 'forum.view,forum.post,forms.escort,forms.bolingbroke,forms.parole,forms.review,admin.view,admin.users,admin.pending'),
(15, 'Administrator', 'Administrator', 15, '*');

-- Insert Forum Categories
INSERT INTO forum_categories (name, description, icon, display_order, required_rank) VALUES
('Announcements', 'Official announcements from Command Staff', 'fa-bullhorn', 1, 1),
('General Discussion', 'General discussion for all members', 'fa-comments', 2, 1),
('FTT Request', 'Field Training Team requests and applications', 'fa-graduation-cap', 3, 1);

-- Insert Administrator User (password: admin123) - Assigned to Operations division
INSERT INTO users (username, ic_name, rank_id, division_id, password, email, badge_number, status) VALUES
('admin', 'Administrator', 15, 1, '$2y$10$CcWdHMe1LKD3I82B4CVUFOMQ6QgVohHOyVnJyllRuCxkxINF7T0He', 'admin@doc.local', 'ADMIN-001', 'active');

-- =====================================================
-- END OF SCHEMA
-- =====================================================
