<?php
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../core/avatar.php';

$action = $_GET['action'] ?? 'view';
$thread_id = (int)($_GET['id'] ?? 0);

// Create new thread
if ($action === 'new' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    requirePermission('forum.post');
    
    $category_id = (int)$_POST['category_id'];
    $title = sanitizeInput($_POST['title']);
    $content = sanitizeInput($_POST['content']);
    $token = $_POST['csrf_token'] ?? '';
    
    if (!verifyToken($token)) {
        die('Invalid token');
    }
    
    if (empty($title) || empty($content)) {
        die('Title and content are required');
    }
    
    // Check if Announcements category (ID 1) requires LT+ (rank level 7+)
    if ($category_id === 1 && ($_SESSION['rank_level'] ?? 0) < 7) {
        die('Only Lieutenant or higher can post announcements');
    }
    
    // Insert thread
    $stmt = $conn->prepare("INSERT INTO threads (category_id, user_id, title) VALUES (?, ?, ?)");
    $stmt->bind_param('iis', $category_id, $_SESSION['user_id'], $title);
    $stmt->execute();
    $thread_id = $conn->insert_id;
    
    // Insert first post
    $stmt = $conn->prepare("INSERT INTO posts (thread_id, user_id, content) VALUES (?, ?, ?)");
    $stmt->bind_param('iis', $thread_id, $_SESSION['user_id'], $content);
    $stmt->execute();
    
    logActivity('thread_create', 'Created thread: ' . $title);
    
    redirect('thread?id=' . $thread_id);
}

// Show new thread form
if ($action === 'new') {
    requirePermission('forum.post');
    
    $category_id = (int)($_GET['category'] ?? 0);
    
    // Get categories user can post to
    $categories_query = "
        SELECT * FROM forum_categories 
        WHERE is_active = 1 AND required_rank <= {$_SESSION['rank_level']}
    ";
    
    // Filter Announcements to LT+ only (rank level 7+)
    $user_rank_level = $_SESSION['rank_level'] ?? 0;
    if ($user_rank_level < 7) {
        $categories_query .= " AND id != 1";
    }
    
    $categories_query .= " ORDER BY display_order";
    $categories = $conn->query($categories_query)->fetch_all(MYSQLI_ASSOC);
    
    $page_title = 'New Thread';
    include __DIR__ . '/../assets/includes/header.php';
    ?>
    
    <div class="page-header">
        <h1>Create New Thread</h1>
    </div>
    
    <form method="POST" class="forum-form">
        <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
        
        <div class="form-group">
            <label for="category_id">Category</label>
            <select name="category_id" id="category_id" required>
                <option value="">Select Category</option>
                <?php foreach ($categories as $cat): ?>
                    <option value="<?= $cat['id'] ?>" <?= $cat['id'] === $category_id ? 'selected' : '' ?>>
                        <?= clean($cat['name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>
        
        <div class="form-group">
            <label for="title">Thread Title</label>
            <input type="text" name="title" id="title" required maxlength="255">
        </div>
        
        <div class="form-group">
            <label for="content">Content</label>
            <textarea name="content" id="content" rows="10" required></textarea>
        </div>
        
        <button type="submit" class="btn btn-primary no-loading">Create Thread</button>
        <a href="forum.php" class="btn btn-secondary">Cancel</a>
    </form>
    
    <?php
    include __DIR__ . '/../assets/includes/footer.php';
    exit;
}

// View thread
if ($thread_id > 0) {
    // Increment view count
    $conn->query("UPDATE threads SET views = views + 1 WHERE id = $thread_id");
    
    // Get thread info
    $thread = $conn->query("
        SELECT t.*, u.ic_name, u.badge_number, u.rank_id, r.display_name as rank_display,
               c.name as category_name, c.id as category_id
        FROM threads t
        JOIN users u ON t.user_id = u.id
        JOIN ranks r ON u.rank_id = r.id
        JOIN forum_categories c ON t.category_id = c.id
        WHERE t.id = $thread_id
    ")->fetch_assoc();
    
    if (!$thread) {
        die('Thread not found');
    }
    
    // Check rank requirement
    $cat_result = $conn->query("SELECT required_rank FROM forum_categories WHERE id = {$thread['category_id']}");
    $required_rank = $cat_result->fetch_assoc()['required_rank'];
    if ($_SESSION['rank_level'] < $required_rank) {
        die('Access denied');
    }
    
    // Handle new post
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$thread['is_locked']) {
        requirePermission('forum.post');
        
        $content = sanitizeInput($_POST['content']);
        $token = $_POST['csrf_token'] ?? '';
        
        if (!verifyToken($token)) {
            die('Invalid token');
        }
        
        if (!empty($content)) {
            $stmt = $conn->prepare("INSERT INTO posts (thread_id, user_id, content) VALUES (?, ?, ?)");
            $stmt->bind_param('iis', $thread_id, $_SESSION['user_id'], $content);
            $stmt->execute();
            
            // Update thread updated_at
            $conn->query("UPDATE threads SET updated_at = NOW() WHERE id = $thread_id");
            
            logActivity('post_create', 'Posted in thread: ' . $thread['title']);
            
            redirect('thread?id=' . $thread_id);
        }
    }
    
    // Get posts with pagination
    $page = max(1, (int)($_GET['page'] ?? 1));
    $per_page = 20;
    
    $result = $conn->query("SELECT COUNT(*) as total FROM posts WHERE thread_id = $thread_id");
    $total = $result->fetch_assoc()['total'];
    $pagination = paginate($total, $per_page, $page);
    
    $posts = $conn->query("
        SELECT p.*, u.ic_name, u.badge_number, u.rank_id, r.display_name as rank_display, r.level as rank_level
        FROM posts p
        JOIN users u ON p.user_id = u.id
        JOIN ranks r ON u.rank_id = r.id
        WHERE p.thread_id = $thread_id
        ORDER BY p.created_at ASC
        LIMIT {$pagination['offset']}, $per_page
    ")->fetch_all(MYSQLI_ASSOC);
    
    $page_title = $thread['title'];
    include __DIR__ . '/../assets/includes/header.php';
    ?>
    
    <div class="breadcrumb">
        <a href="forum.php">Forum</a> » 
        <a href="forum.php?category=<?= $thread['category_id'] ?>"><?= clean($thread['category_name']) ?></a> »
        <?= clean($thread['title']) ?>
    </div>
    
    <div class="thread-header">
        <div class="thread-info">
            <?php if ($thread['is_pinned']): ?>
                <span class="badge badge-warning">📌 Pinned</span>
            <?php endif; ?>
            <?php if ($thread['is_locked']): ?>
                <span class="badge badge-secondary">🔒 Locked</span>
            <?php endif; ?>
            <h1><?= clean($thread['title']) ?></h1>
            <div class="thread-meta">
                Started by <strong><?= clean($thread['ic_name']) ?></strong> • 
                <?= formatDate($thread['created_at']) ?> • 
                <?= $thread['views'] ?> views
            </div>
        </div>
        
        <?php if (hasPermission('forum.moderate')): ?>
            <div class="thread-actions">
                <a href="post.php?action=pin&id=<?= $thread_id ?>" class="btn btn-sm">📌 Pin/Unpin</a>
                <a href="post.php?action=lock&id=<?= $thread_id ?>" class="btn btn-sm">🔒 Lock/Unlock</a>
            </div>
        <?php endif; ?>
    </div>
    
    <div class="posts-container">
        <?php foreach ($posts as $post): ?>
            <div class="post-card" id="post-<?= $post['id'] ?>">
                <div class="post-author">
                    <img src="<?= getAvatarByRank($post['rank_id']) ?>" alt="Avatar" class="author-avatar">
                    <div class="author-info">
                        <strong><?= clean($post['ic_name']) ?></strong>
                        <span class="rank-badge" style="background: <?= getRankColor($post['rank_level']) ?>">
                            <?= clean($post['rank_display']) ?>
                        </span>
                        <?php if ($post['badge_number']): ?>
                            <span class="badge-number">#<?= clean($post['badge_number']) ?></span>
                        <?php endif; ?>
                    </div>
                </div>
                <div class="post-content">
                    <div class="post-text">
                        <?= nl2br(clean($post['content'])) ?>
                    </div>
                    <div class="post-footer">
                        <span class="post-time"><?= formatDate($post['created_at']) ?></span>
                        <?php if ($post['is_edited']): ?>
                            <span class="edited-tag">Edited <?= formatDate($post['edited_at']) ?></span>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
    
    <?php if ($pagination['total_pages'] > 1): ?>
        <div class="pagination">
            <?php for ($i = 1; $i <= $pagination['total_pages']; $i++): ?>
                <a href="?id=<?= $thread_id ?>&page=<?= $i ?>" 
                   class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
            <?php endfor; ?>
        </div>
    <?php endif; ?>
    
    <?php if (!$thread['is_locked'] && hasPermission('forum.post')): ?>
        <div class="reply-form">
            <h3>Post Reply</h3>
            <form method="POST">
                <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                <div class="form-group">
                    <textarea name="content" rows="6" placeholder="Write your reply..." required></textarea>
                </div>
                <button type="submit" class="btn btn-primary no-loading">Post Reply</button>
            </form>
        </div>
    <?php elseif ($thread['is_locked']): ?>
        <div class="alert alert-warning">
            🔒 This thread is locked. No new replies can be posted.
        </div>
    <?php endif; ?>
    
    <?php
    include __DIR__ . '/../assets/includes/footer.php';
}
?>