<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../core/session.php';
require_once __DIR__ . '/../core/functions.php';
require_once __DIR__ . '/../core/rbac.php';

// Redirect if already logged in
if (validateSession()) {
    redirect('dashboard');
}

$error = '';
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $token = $_POST['csrf_token'] ?? '';
    if (!verifyToken($token)) {
        $error = 'Invalid security token';
    } else {
        $username = sanitizeInput($_POST['username'] ?? '');
        $ic_name = sanitizeInput($_POST['ic_name'] ?? '');
        $badge_number = sanitizeInput($_POST['badge_number'] ?? '');
        $email = sanitizeInput($_POST['email'] ?? '');
        $password = $_POST['password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        
        // Validation
        if (empty($username) || empty($ic_name) || empty($password) || empty($confirm_password)) {
            $error = 'All required fields must be filled';
        } elseif (strlen($username) < 3) {
            $error = 'Username must be at least 3 characters';
        } elseif (strlen($password) < 6) {
            $error = 'Password must be at least 6 characters';
        } elseif ($password !== $confirm_password) {
            $error = 'Passwords do not match';
        } else {
            // Check if username already exists
            $stmt = $conn->prepare("SELECT id FROM users WHERE username = ?");
            $stmt->bind_param('s', $username);
            $stmt->execute();
            $result = $stmt->get_result();
            
            if ($result->num_rows > 0) {
                $error = 'Username already exists';
            } else {
                // Check if email already exists (if provided)
                if (!empty($email)) {
                    $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
                    $stmt->bind_param('s', $email);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    
                    if ($result->num_rows > 0) {
                        $error = 'Email already registered';
                    }
                }
                
                if (empty($error)) {
                    // Hash password
                    $password_hash = password_hash($password, PASSWORD_DEFAULT);
                    
                    // Get Trainee rank (level 1)
                    $rank_result = $conn->query("SELECT id FROM ranks WHERE level = 1 LIMIT 1");
                    $rank_id = $rank_result->fetch_assoc()['id'];
                    
                    // Insert new user with pending status and default Operations division
                    $status = 'pending'; // Pending approval
                    $default_division = 1; // Operations
                    
                    $stmt = $conn->prepare("
                        INSERT INTO users (username, ic_name, badge_number, email, password, rank_id, division_id, status) 
                        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
                    ");
                    $stmt->bind_param('sssssiis', $username, $ic_name, $badge_number, $email, $password_hash, $rank_id, $default_division, $status);
                    
                    if ($stmt->execute()) {
                        // Log registration
                        $user_id = $conn->insert_id;
                        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
                        $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
                        
                        $stmt = $conn->prepare("
                            INSERT INTO activity_logs (user_id, action, description, ip_address, user_agent) 
                            VALUES (?, 'registration', 'New user registration - pending approval', ?, ?)
                        ");
                        $stmt->bind_param('iss', $user_id, $ip, $user_agent);
                        $stmt->execute();
                        
                        $success = 'Registration successful! Your account is pending approval. You will be able to login once an administrator approves your account.';
                    } else {
                        $error = 'Registration failed: ' . $stmt->error . '. Please contact administrator.';
                    }
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - DOC IC Forum</title>
    <!-- Prevent flash of light theme -->
    <script>
        // Apply theme IMMEDIATELY before CSS loads to prevent flash
        (function() {
            const theme = localStorage.getItem('doc-theme');
            if (theme === 'dark') {
                document.documentElement.classList.add('dark-theme');
            }
        })();
    </script>
    <link rel="stylesheet" href="/doc/assets/css/style.css">
</head>
<body class="login-page">
    <!-- Theme Toggle Button for Register Page -->
    <button id="theme-toggle" class="btn btn-sm" style="position: fixed; top: 20px; right: 20px; z-index: 9999;" title="Toggle Dark/Light Mode">
        <span id="theme-icon">🌙</span>
    </button>
    
    <div class="login-container">
        <div class="login-box">
            <div class="login-header">
                <h1>Department Of Corrections</h1>
                <p>Forums</p>
                <p class="motto">To Secure, To Rehabilitate, To Reintegrate</p>
            </div>
            
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <?= clean($error) ?>
                </div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-info">
                    <?= clean($success) ?>
                </div>
                <div style="padding: 20px 30px;">
                    <a href="/doc/login" class="btn btn-primary btn-block">Back to Login</a>
                </div>
            <?php else: ?>
            
            <form method="POST" class="login-form">
                <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                
                <div class="form-group">
                    <label for="username">Username *</label>
                    <input type="text" id="username" name="username" required minlength="3" 
                           value="<?= clean($_POST['username'] ?? '') ?>" autofocus>
                    <small>Minimum 3 characters</small>
                </div>
                
                <div class="form-group">
                    <label for="ic_name">IC Name (In-Character) *</label>
                    <input type="text" id="ic_name" name="ic_name" required 
                           value="<?= clean($_POST['ic_name'] ?? '') ?>">
                    <small>Your character name for roleplay</small>
                </div>
                
                <div class="form-group">
                    <label for="badge_number">Badge Number (Optional)</label>
                    <input type="text" id="badge_number" name="badge_number" 
                           value="<?= clean($_POST['badge_number'] ?? '') ?>" 
                           placeholder="e.g., OFF-001">
                </div>
                
                <div class="form-group">
                    <label for="email">Email (Optional)</label>
                    <input type="email" id="email" name="email" 
                           value="<?= clean($_POST['email'] ?? '') ?>">
                </div>
                
                <div class="form-group">
                    <label for="password">Password *</label>
                    <input type="password" id="password" name="password" required minlength="6">
                    <small>Minimum 6 characters</small>
                </div>
                
                <div class="form-group">
                    <label for="confirm_password">Confirm Password *</label>
                    <input type="password" id="confirm_password" name="confirm_password" required minlength="6">
                </div>
                
                <button type="submit" class="btn btn-primary btn-block no-loading">Register</button>
                
                <div style="text-align: center; margin-top: 15px;">
                    <a href="/doc/login" style="color: var(--primary); text-decoration: none;">
                        Already have an account? Login here
                    </a>
                </div>
            </form>
            
            <?php endif; ?>
            
            <div class="login-footer">
                <p>&copy; <?= date('Y') ?> Departement Of Corrections iMe. All rights reserved.</p>
            </div>
        </div>
    </div>
    
    <script src="/doc/assets/js/app.js"></script>
</body>
</html>
