<?php
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../core/session.php';
require_once __DIR__ . '/../core/functions.php';
require_once __DIR__ . '/../core/rbac.php';
require_once __DIR__ . '/../core/avatar.php';

// Redirect if already logged in
if (validateSession()) {
    redirect('dashboard');
}

$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = sanitizeInput($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    
    if (empty($username) || empty($password)) {
        $error = 'Please enter both username and password';
    } else {
        // Get user from database
        $stmt = $conn->prepare("
            SELECT u.*, r.name as rank_name, r.display_name, r.level as rank_level, r.permissions, d.code as division_code, d.name as division_name
            FROM users u 
            JOIN ranks r ON u.rank_id = r.id 
            LEFT JOIN divisions d ON u.division_id = d.id
            WHERE u.username = ?
        ");
        $stmt->bind_param('s', $username);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($user = $result->fetch_assoc()) {
            if (password_verify($password, $user['password'])) {
                // Check if account is pending approval
                if ($user['status'] === 'pending') {
                    $error = 'Your account is pending approval. Please wait for an administrator to approve your registration.';
                } elseif ($user['status'] !== 'active') {
                    $error = 'Your account has been ' . $user['status'] . '. Please contact an administrator.';
                } else {
                    // Set session variables
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['username'] = $user['username'];
                    $_SESSION['ic_name'] = $user['ic_name'];
                    $_SESSION['rank_id'] = $user['rank_id'];
                    $_SESSION['rank_name'] = $user['rank_name'];
                    $_SESSION['rank_display'] = $user['display_name'];
                    $_SESSION['rank_level'] = $user['rank_level'];
                    $_SESSION['badge_number'] = $user['badge_number'];
                    $_SESSION['avatar'] = getAvatarByRank($user['rank_id']); // Avatar based on rank
                    $_SESSION['division_id'] = $user['division_id'];
                    $_SESSION['division_code'] = $user['division_code'] ?? 'OPS';
                    $_SESSION['division_name'] = $user['division_name'] ?? 'Operations';
                    $_SESSION['permissions'] = explode(',', $user['permissions'] ?? '');
                    $_SESSION['last_activity'] = time();
                    
                    // Update last login
                    $stmt = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                    $stmt->bind_param('i', $user['id']);
                    $stmt->execute();
                    
                    // Log activity
                    logActivity('login', 'User logged in');
                    
                    redirect('dashboard');
                }
            } else {
                $error = 'Invalid username or password';
            }
        } else {
            $error = 'Invalid username or password';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Department Of Corrections</title>
    <!-- Prevent flash of light theme -->
    <script>
        // Apply theme IMMEDIATELY before CSS loads to prevent flash
        (function() {
            const theme = localStorage.getItem('doc-theme');
            if (theme === 'dark') {
                document.documentElement.classList.add('dark-theme');
            }
        })();
    </script>
    <link rel="stylesheet" href="/doc/assets/css/style.css">
</head>
<body class="login-page">
    <!-- Theme Toggle Button for Login Page -->
    <button id="theme-toggle" class="btn btn-sm" style="position: fixed; top: 20px; right: 20px; z-index: 9999;" title="Toggle Dark/Light Mode">
        <span id="theme-icon">🌙</span>
    </button>
    
    <div class="login-container">
        <div class="login-box">
            <div class="login-header">
                <h1>Department Of Corrections </h1>
                <p>Forums</p>
                <p class="motto">To Secure, To Rehabilitate, To Reintegrate</p>
            </div>
            
            <?php if ($error): ?>
                <div class="alert alert-error">
                    <?= clean($error) ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" class="login-form">
                <div class="form-group">
                    <label for="username">Username</label>
                    <input type="text" id="username" name="username" required autofocus 
                           value="<?= clean($_POST['username'] ?? '') ?>">
                </div>
                
                <div class="form-group">
                    <label for="password">Password</label>
                    <input type="password" id="password" name="password" required>
                </div>
                
                <button type="submit" class="btn btn-primary btn-block no-loading">Login</button>
                
                <div style="text-align: center; margin-top: 15px;">
                    <a href="/doc/register" style="color: var(--primary); text-decoration: none; font-weight: 600;">
                        Don't have an account? Register here
                    </a>
                </div>
            </form>
            
            <div class="login-footer">
                <p>&copy; <?= date('Y') ?> Departement Of Corrections iMe. All rights reserved.</p>
            </div>
        </div>
    </div>
    
    <script src="/doc/assets/js/app.js"></script>
</body>
</html>