<?php
require_once __DIR__ . '/../config/auth.php';

// Check access: Major ADMIN (rank_id 11) or Commaderi (rank_id 9 - Commander PSB) or Deputy Chief+ (level 13+)
$rank_id = $_SESSION['rank_id'] ?? 0;
$rank_level = $_SESSION['rank_level'] ?? 0;

if ($rank_id != 9 && $rank_id != 11 && $rank_level < 13) {
    die('Access denied. Major ADMIN, Commander PSB or Deputy Chief Warden+ only.');
}

// Can issue strike points if in IA division or rank 9/11
$division_code = $_SESSION['division_code'] ?? '';
$is_ia = ($division_code === 'IA' || $rank_id == 9 || $rank_id == 11);

$action = $_GET['action'] ?? 'list';
$id = (int)($_GET['id'] ?? 0);
$page = max(1, (int)($_GET['page'] ?? 1));
$per_page = 20;

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    validateToken();
    
    $post_action = $_POST['action'] ?? '';
    
    // Issue strike point
    if ($post_action === 'issue' && $is_ia) {
        $target_user_id = (int)$_POST['target_user_id'];
        $points = min(3, max(1, (int)$_POST['points'])); // 1-3 points only
        $reason = sanitizeInput($_POST['reason']);
        $incident_date = $_POST['incident_date'];
        $evidence_link = sanitizeInput($_POST['evidence_link']);
        $expires_at = $_POST['expires_at'];
        $notes = sanitizeInput($_POST['notes']);
        
        $stmt = $conn->prepare("
            INSERT INTO strike_points 
            (target_user_id, issued_by, points, reason, incident_date, evidence_link, expires_at, notes) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->bind_param('iiisssss', $target_user_id, $_SESSION['user_id'], $points, $reason, $incident_date, $evidence_link, $expires_at, $notes);
        $stmt->execute();
        
        logActivity('ia_strike', "Issued $points strike point(s) to user ID $target_user_id");
        redirect('ia');
    }
    
    // Update strike status
    if ($post_action === 'update_status' && $id > 0) {
        $new_status = $_POST['status'];
        $notes = sanitizeInput($_POST['notes']);
        
        $stmt = $conn->prepare("UPDATE strike_points SET status = ?, notes = CONCAT(notes, '\n\n[Update] ', ?) WHERE id = ?");
        $stmt->bind_param('ssi', $new_status, $notes, $id);
        $stmt->execute();
        
        logActivity('ia_strike', "Updated strike point #$id status to $new_status");
        redirect('ia?action=view&id=' . $id);
    }
}

$page_title = 'Internal Affairs - Strike Point Logs';
include __DIR__ . '/../assets/includes/header.php';
?>

<div class="page-header">
    <h1>🔍 Internal Affairs - Strike Point Logs</h1>
    <div>
        <a href="/doc/dashboard" class="btn btn-secondary">← Dashboard</a>
        <?php if ($is_ia): ?>
            <a href="?action=new" class="btn btn-primary">+ Issue Strike Point</a>
        <?php endif; ?>
    </div>
</div>

<?php if ($action === 'new' && $is_ia): ?>
    <div class="doc-form">
        <h2>Issue Strike Point</h2>
        <form method="POST">
            <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
            <input type="hidden" name="action" value="issue">
            
            <div class="form-row">
                <div class="form-group">
                    <label for="target_user_id">Target Officer *</label>
                    <select name="target_user_id" id="target_user_id" required>
                        <option value="">Select Officer...</option>
                        <?php
                        $users = $conn->query("
                            SELECT u.id, u.ic_name, u.badge_number, r.display_name as rank
                            FROM users u
                            JOIN ranks r ON u.rank_id = r.id
                            WHERE u.status = 'active'
                            ORDER BY u.ic_name
                        ")->fetch_all(MYSQLI_ASSOC);
                        foreach ($users as $user): 
                        ?>
                            <option value="<?= $user['id'] ?>">
                                <?= clean($user['ic_name']) ?> (<?= clean($user['badge_number']) ?>) - <?= clean($user['rank']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="points">Strike Points *</label>
                    <select name="points" id="points" required>
                        <option value="1">1 Point - Minor Violation</option>
                        <option value="2">2 Points - Moderate Violation</option>
                        <option value="3">3 Points - Major Violation</option>
                    </select>
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label for="incident_date">Incident Date/Time *</label>
                    <input type="datetime-local" name="incident_date" id="incident_date" required>
                </div>
                
                <div class="form-group">
                    <label for="expires_at">Expiration Date</label>
                    <input type="date" name="expires_at" id="expires_at" 
                           title="Leave empty for no expiration">
                </div>
            </div>
            
            <div class="form-group">
                <label for="reason">Violation Reason *</label>
                <textarea name="reason" id="reason" rows="4" required 
                          placeholder="Describe the violation..."></textarea>
            </div>
            
            <div class="form-group">
                <label for="evidence_link">Evidence Link</label>
                <input type="url" name="evidence_link" id="evidence_link" 
                       placeholder="https://imgur.com/... or video link">
            </div>
            
            <div class="form-group">
                <label for="notes">Additional Notes</label>
                <textarea name="notes" id="notes" rows="3" 
                          placeholder="Any additional information..."></textarea>
            </div>
            
            <button type="submit" class="btn btn-primary no-loading">Issue Strike Point</button>
            <a href="ia.php" class="btn btn-secondary">Cancel</a>
        </form>
    </div>

<?php elseif ($action === 'view' && $id > 0): ?>
    <?php
    $strike = $conn->query("
        SELECT sp.*, 
               target.ic_name as target_name, target.badge_number as target_badge,
               issuer.ic_name as issuer_name, issuer.badge_number as issuer_badge,
               tr.display_name as target_rank, ir.display_name as issuer_rank
        FROM strike_points sp
        JOIN users target ON sp.target_user_id = target.id
        JOIN users issuer ON sp.issued_by = issuer.id
        JOIN ranks tr ON target.rank_id = tr.id
        JOIN ranks ir ON issuer.rank_id = ir.id
        WHERE sp.id = $id
    ")->fetch_assoc();
    
    if (!$strike) die('Strike point not found');
    ?>
    
    <div class="doc-form">
        <h2>Strike Point #<?= $strike['id'] ?></h2>
        
        <div class="form-section">
            <h3>Target Officer</h3>
            <p><strong><?= clean($strike['target_name']) ?></strong> (<?= clean($strike['target_badge']) ?>)</p>
            <p>Rank: <?= clean($strike['target_rank']) ?></p>
        </div>
        
        <div class="form-section">
            <h3>Strike Details</h3>
            <p><strong>Points:</strong> <?= $strike['points'] ?> Point(s)</p>
            <p><strong>Status:</strong> <span class="badge badge-<?= $strike['status'] ?>"><?= ucfirst($strike['status']) ?></span></p>
            <p><strong>Incident Date:</strong> <?= formatDate($strike['incident_date']) ?></p>
            <?php if ($strike['expires_at']): ?>
                <p><strong>Expires:</strong> <?= date('M d, Y', strtotime($strike['expires_at'])) ?></p>
            <?php endif; ?>
        </div>
        
        <div class="form-section">
            <h3>Violation Reason</h3>
            <p><?= nl2br(clean($strike['reason'])) ?></p>
        </div>
        
        <?php if ($strike['evidence_link']): ?>
            <div class="form-section">
                <h3>Evidence</h3>
                <p><a href="<?= clean($strike['evidence_link']) ?>" target="_blank" class="btn btn-sm">View Evidence</a></p>
            </div>
        <?php endif; ?>
        
        <?php if ($strike['notes']): ?>
            <div class="form-section">
                <h3>Notes</h3>
                <p><?= nl2br(clean($strike['notes'])) ?></p>
            </div>
        <?php endif; ?>
        
        <div class="form-section">
            <h3>Issued By</h3>
            <p><strong><?= clean($strike['issuer_name']) ?></strong> (<?= clean($strike['issuer_badge']) ?>)</p>
            <p>Rank: <?= clean($strike['issuer_rank']) ?></p>
            <p>Date: <?= formatDate($strike['created_at']) ?></p>
        </div>
        
        <?php if ($is_ia && $strike['status'] === 'active'): ?>
            <div class="form-section">
                <h3>Update Status</h3>
                <form method="POST" style="display: inline-block; margin-right: 10px;">
                    <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                    <input type="hidden" name="action" value="update_status">
                    <input type="hidden" name="status" value="appealed">
                    <input type="text" name="notes" placeholder="Appeal notes..." required style="margin-right: 5px;">
                    <button type="submit" class="btn btn-sm no-loading">Mark as Appealed</button>
                </form>
                
                <form method="POST" style="display: inline-block; margin-right: 10px;">
                    <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                    <input type="hidden" name="action" value="update_status">
                    <input type="hidden" name="status" value="dismissed">
                    <input type="text" name="notes" placeholder="Dismissal reason..." required style="margin-right: 5px;">
                    <button type="submit" class="btn btn-sm btn-secondary no-loading">Dismiss</button>
                </form>
                
                <form method="POST" style="display: inline-block;">
                    <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                    <input type="hidden" name="action" value="update_status">
                    <input type="hidden" name="status" value="expired">
                    <input type="text" name="notes" placeholder="Expiration notes..." style="margin-right: 5px;">
                    <button type="submit" class="btn btn-sm btn-secondary no-loading">Mark as Expired</button>
                </form>
            </div>
        <?php endif; ?>
        
        <a href="ia.php" class="btn btn-secondary">← Back to List</a>
    </div>

<?php else: // List view ?>
    <?php
    $result = $conn->query("SELECT COUNT(*) as total FROM strike_points");
    $total = $result->fetch_assoc()['total'];
    $pagination = paginate($total, $per_page, $page);
    
    $strikes = $conn->query("
        SELECT sp.*, 
               u.ic_name as target_name, u.badge_number as target_badge,
               issuer.ic_name as issuer_name,
               r.display_name as target_rank
        FROM strike_points sp
        JOIN users u ON sp.target_user_id = u.id
        JOIN users issuer ON sp.issued_by = issuer.id
        JOIN ranks r ON u.rank_id = r.id
        ORDER BY sp.created_at DESC
        LIMIT {$pagination['offset']}, $per_page
    ")->fetch_all(MYSQLI_ASSOC);
    ?>
    
    <table class="data-table">
        <thead>
            <tr>
                <th>ID</th>
                <th>Target Officer</th>
                <th>Points</th>
                <th>Reason</th>
                <th>Incident Date</th>
                <th>Status</th>
                <th>Issued By</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($strikes as $strike): ?>
                <tr>
                    <td>#<?= $strike['id'] ?></td>
                    <td>
                        <strong><?= clean($strike['target_name']) ?></strong><br>
                        <small><?= clean($strike['target_badge']) ?> - <?= clean($strike['target_rank']) ?></small>
                    </td>
                    <td>
                        <span class="badge badge-<?= $strike['points'] === 3 ? 'danger' : ($strike['points'] === 2 ? 'warning' : 'secondary') ?>">
                            <?= $strike['points'] ?> Point(s)
                        </span>
                    </td>
                    <td><?= substr(clean($strike['reason']), 0, 80) ?>...</td>
                    <td><?= formatDate($strike['incident_date']) ?></td>
                    <td><span class="badge badge-<?= $strike['status'] ?>"><?= ucfirst($strike['status']) ?></span></td>
                    <td><?= clean($strike['issuer_name']) ?></td>
                    <td>
                        <a href="?action=view&id=<?= $strike['id'] ?>" class="btn btn-sm">View</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
    
    <?php if ($pagination['total_pages'] > 1): ?>
        <div class="pagination">
            <?php for ($i = 1; $i <= $pagination['total_pages']; $i++): ?>
                <a href="?page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
            <?php endfor; ?>
        </div>
    <?php endif; ?>
<?php endif; ?>

<?php include __DIR__ . '/../assets/includes/footer.php'; ?>
