<?php
require_once __DIR__ . '/../config/auth.php';

// Get all categories with thread counts
$categories = $conn->query("
    SELECT c.*, 
           COUNT(DISTINCT t.id) as thread_count,
           COUNT(DISTINCT p.id) as post_count,
           (SELECT t2.title FROM threads t2 WHERE t2.category_id = c.id ORDER BY t2.updated_at DESC LIMIT 1) as last_thread_title,
           (SELECT t2.id FROM threads t2 WHERE t2.category_id = c.id ORDER BY t2.updated_at DESC LIMIT 1) as last_thread_id,
           (SELECT u.ic_name FROM threads t2 JOIN users u ON t2.user_id = u.id WHERE t2.category_id = c.id ORDER BY t2.updated_at DESC LIMIT 1) as last_author
    FROM forum_categories c
    LEFT JOIN threads t ON c.id = t.category_id
    LEFT JOIN posts p ON t.id = p.thread_id
    WHERE c.is_active = 1 AND c.required_rank <= {$_SESSION['rank_level']}
    GROUP BY c.id
    ORDER BY c.display_order, c.name
")->fetch_all(MYSQLI_ASSOC);

$page_title = 'Forum';
include __DIR__ . '/../assets/includes/header.php';
?>

<div class="page-header">
    <h1>📋 Forum Categories</h1>
    <?php if (hasPermission('forum.post')): ?>
        <a href="thread.php?action=new" class="btn btn-primary">+ New Thread</a>
    <?php endif; ?>
</div>

<div class="forum-categories">
    <?php foreach ($categories as $category): ?>
        <div class="category-card">
            <div class="category-icon">
                <?= $category['icon'] ? $category['icon'] : '📁' ?>
            </div>
            <div class="category-info">
                <h2>
                    <a href="forum.php?category=<?= $category['id'] ?>">
                        <?= clean($category['name']) ?>
                    </a>
                </h2>
                <p><?= clean($category['description']) ?></p>
                <?php if ($category['last_thread_title']): ?>
                    <div class="last-post">
                        <strong>Last:</strong>
                        <a href="thread.php?id=<?= $category['last_thread_id'] ?>">
                            <?= clean($category['last_thread_title']) ?>
                        </a>
                        by <?= clean($category['last_author']) ?>
                    </div>
                <?php endif; ?>
            </div>
            <div class="category-stats">
                <div class="stat">
                    <span class="count"><?= $category['thread_count'] ?></span>
                    <span class="label">Threads</span>
                </div>
                <div class="stat">
                    <span class="count"><?= $category['post_count'] ?></span>
                    <span class="label">Posts</span>
                </div>
            </div>
        </div>
    <?php endforeach; ?>
    
    <?php if (empty($categories)): ?>
        <p class="no-data">No forum categories available</p>
    <?php endif; ?>
</div>

<?php
// If category is selected, show threads
if (isset($_GET['category'])) {
    $category_id = (int)$_GET['category'];
    $page = max(1, (int)($_GET['page'] ?? 1));
    $per_page = 20;
    
    // Get category info
    $stmt = $conn->prepare("SELECT * FROM forum_categories WHERE id = ? AND required_rank <= ?");
    $stmt->bind_param('ii', $category_id, $_SESSION['rank_level']);
    $stmt->execute();
    $category = $stmt->get_result()->fetch_assoc();
    
    if ($category) {
        // Get total threads
        $result = $conn->query("SELECT COUNT(*) as total FROM threads WHERE category_id = $category_id");
        $total = $result->fetch_assoc()['total'];
        $pagination = paginate($total, $per_page, $page);
        
        // Get threads
        $threads = $conn->query("
            SELECT t.*, u.ic_name, u.badge_number, u.rank_id, r.display_name as rank_display,
                   (SELECT COUNT(*) FROM posts WHERE thread_id = t.id) as post_count,
                   (SELECT u2.ic_name FROM posts p2 JOIN users u2 ON p2.user_id = u2.id WHERE p2.thread_id = t.id ORDER BY p2.created_at DESC LIMIT 1) as last_poster
            FROM threads t
            JOIN users u ON t.user_id = u.id
            JOIN ranks r ON u.rank_id = r.id
            WHERE t.category_id = $category_id
            ORDER BY t.is_pinned DESC, t.updated_at DESC
            LIMIT {$pagination['offset']}, $per_page
        ")->fetch_all(MYSQLI_ASSOC);
        ?>
        
        <div class="category-threads">
            <div class="section-header">
                <h2><?= clean($category['name']) ?> - Threads</h2>
                <?php if (hasPermission('forum.post')): ?>
                    <a href="thread.php?action=new&category=<?= $category_id ?>" class="btn btn-primary">+ New Thread</a>
                <?php endif; ?>
            </div>
            
            <div class="threads-table">
                <?php foreach ($threads as $thread): ?>
                    <div class="thread-row">
                        <div class="thread-main">
                            <?php if ($thread['is_pinned']): ?>
                                <span class="badge badge-warning">📌</span>
                            <?php endif; ?>
                            <?php if ($thread['is_locked']): ?>
                                <span class="badge badge-secondary">🔒</span>
                            <?php endif; ?>
                            <h3>
                                <a href="thread.php?id=<?= $thread['id'] ?>">
                                    <?= clean($thread['title']) ?>
                                </a>
                            </h3>
                            <div class="thread-meta">
                                Started by <strong><?= clean($thread['ic_name']) ?></strong>
                                <span class="rank-badge" style="background: <?= getRankColor($thread['rank_id']) ?>">
                                    <?= clean($thread['rank_display']) ?>
                                </span>
                                • <?= timeAgo($thread['created_at']) ?>
                            </div>
                        </div>
                        <div class="thread-stats">
                            <span>💬 <?= $thread['post_count'] ?></span>
                            <span>👁 <?= $thread['views'] ?></span>
                        </div>
                    </div>
                <?php endforeach; ?>
                
                <?php if (empty($threads)): ?>
                    <p class="no-data">No threads in this category yet</p>
                <?php endif; ?>
            </div>
            
            <?php if ($pagination['total_pages'] > 1): ?>
                <div class="pagination">
                    <?php for ($i = 1; $i <= $pagination['total_pages']; $i++): ?>
                        <a href="?category=<?= $category_id ?>&page=<?= $i ?>" 
                           class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
                    <?php endfor; ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
}
?>

<?php include __DIR__ . '/../assets/includes/footer.php'; ?>