<?php
require_once __DIR__ . '/../config/auth.php';

// Check access: majora (rank_id 12 - Major DOFO) or Deputy Chief+ (level 13+)
$rank_id = $_SESSION['rank_id'] ?? 0;
$rank_level = $_SESSION['rank_level'] ?? 0;

if ($rank_id != 12 && $rank_level < 13) {
    die('Access denied. Major DOFO or Deputy Chief Warden+ only.');
}

// Permission checks
$division_code = $_SESSION['division_code'] ?? '';
$is_ctd = ($division_code === 'CTD');
$is_major_plus = ($rank_id == 12 || $rank_level >= 13); // Major DOFO or Deputy Chief+

$channel = $_GET['channel'] ?? 'SRU';
$action = $_GET['action'] ?? 'list';
$id = (int)($_GET['id'] ?? 0);

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    validateToken();
    
    $post_action = $_POST['action'] ?? '';
    
    // Create mission (Major+ only)
    if ($post_action === 'create' && $is_major_plus) {
        $mission_channel = $_POST['channel'];
        $title = sanitizeInput($_POST['title']);
        $description = sanitizeInput($_POST['description']);
        $assigned_to = sanitizeInput($_POST['assigned_to']);
        $priority = $_POST['priority'];
        $due_date = $_POST['due_date'];
        
        $stmt = $conn->prepare("
            INSERT INTO ctd_missions 
            (channel, title, description, assigned_by, assigned_to, priority, due_date) 
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->bind_param('sssisss', $mission_channel, $title, $description, $_SESSION['user_id'], $assigned_to, $priority, $due_date);
        $stmt->execute();
        
        logActivity('ctd_mission', "Created $mission_channel mission: $title");
        redirect("ctd?channel=$mission_channel");
    }
    
    // Update mission status
    if ($post_action === 'update' && $id > 0) {
        $new_status = $_POST['status'];
        $completion_notes = sanitizeInput($_POST['completion_notes']);
        
        if ($new_status === 'completed') {
            $stmt = $conn->prepare("
                UPDATE ctd_missions 
                SET status = ?, completion_notes = ?, completed_at = NOW() 
                WHERE id = ?
            ");
            $stmt->bind_param('ssi', $new_status, $completion_notes, $id);
        } else {
            $stmt = $conn->prepare("UPDATE ctd_missions SET status = ? WHERE id = ?");
            $stmt->bind_param('si', $new_status, $id);
        }
        $stmt->execute();
        
        logActivity('ctd_mission', "Updated mission #$id status to $new_status");
        redirect("ctd?action=view&id=$id&channel=$channel");
    }
}

$page_title = 'Correctional Tactical Division';
include __DIR__ . '/../assets/includes/header.php';
?>

<div class="page-header">
    <h1>🎯 Correctional Tactical Division</h1>
    <div>
        <a href="/doc/dashboard" class="btn btn-secondary">← Dashboard</a>
        <?php if ($is_major_plus): ?>
            <a href="?action=new&channel=<?= $channel ?>" class="btn btn-primary">+ Create Mission</a>
        <?php endif; ?>
    </div>
</div>

<div class="admin-tabs">
    <a href="?channel=SRU" class="tab <?= $channel === 'SRU' ? 'active' : '' ?>">🚨 Special Response Unit</a>
    <a href="?channel=BEA" class="tab <?= $channel === 'BEA' ? 'active' : '' ?>">⚖️ Bail Enforcer Agent</a>
</div>

<?php if ($action === 'new' && $is_major_plus): ?>
    <div class="doc-form">
        <h2>Create <?= $channel ?> Mission</h2>
        <form method="POST">
            <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
            <input type="hidden" name="action" value="create">
            <input type="hidden" name="channel" value="<?= $channel ?>">
            
            <div class="form-row">
                <div class="form-group">
                    <label for="title">Mission Title *</label>
                    <input type="text" name="title" id="title" required>
                </div>
                
                <div class="form-group">
                    <label for="priority">Priority *</label>
                    <select name="priority" id="priority" required>
                        <option value="low">Low</option>
                        <option value="medium" selected>Medium</option>
                        <option value="high">High</option>
                        <option value="critical">Critical</option>
                    </select>
                </div>
            </div>
            
            <div class="form-group">
                <label for="description">Mission Description *</label>
                <textarea name="description" id="description" rows="6" required 
                          placeholder="Describe the mission objectives..."></textarea>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label for="assigned_to">Assigned To</label>
                    <input type="text" name="assigned_to" id="assigned_to" 
                           placeholder="Officer names (comma separated)">
                </div>
                
                <div class="form-group">
                    <label for="due_date">Due Date</label>
                    <input type="datetime-local" name="due_date" id="due_date">
                </div>
            </div>
            
            <button type="submit" class="btn btn-primary no-loading">Create Mission</button>
            <a href="?channel=<?= $channel ?>" class="btn btn-secondary">Cancel</a>
        </form>
    </div>

<?php elseif ($action === 'view' && $id > 0): ?>
    <?php
    $mission = $conn->query("
        SELECT m.*, u.ic_name as assigned_by_name, u.badge_number, r.display_name as rank
        FROM ctd_missions m
        JOIN users u ON m.assigned_by = u.id
        JOIN ranks r ON u.rank_id = r.id
        WHERE m.id = $id
    ")->fetch_assoc();
    
    if (!$mission) die('Mission not found');
    ?>
    
    <div class="doc-form">
        <h2><?= clean($mission['title']) ?></h2>
        
        <div class="form-section">
            <p>
                <strong>Channel:</strong> 
                <span class="badge badge-primary"><?= $mission['channel'] ?></span>
            </p>
            <p>
                <strong>Priority:</strong> 
                <span class="badge badge-<?= $mission['priority'] === 'critical' ? 'danger' : ($mission['priority'] === 'high' ? 'warning' : 'secondary') ?>">
                    <?= ucfirst($mission['priority']) ?>
                </span>
            </p>
            <p>
                <strong>Status:</strong> 
                <span class="badge badge-<?= $mission['status'] ?>">
                    <?= ucfirst(str_replace('_', ' ', $mission['status'])) ?>
                </span>
            </p>
        </div>
        
        <div class="form-section">
            <h3>Mission Description</h3>
            <p><?= nl2br(clean($mission['description'])) ?></p>
        </div>
        
        <?php if ($mission['assigned_to']): ?>
            <div class="form-section">
                <h3>Assigned To</h3>
                <p><?= clean($mission['assigned_to']) ?></p>
            </div>
        <?php endif; ?>
        
        <div class="form-section">
            <h3>Mission Details</h3>
            <p><strong>Assigned By:</strong> <?= clean($mission['assigned_by_name']) ?> (<?= clean($mission['badge_number']) ?>) - <?= clean($mission['rank']) ?></p>
            <p><strong>Created:</strong> <?= formatDate($mission['created_at']) ?></p>
            <?php if ($mission['due_date']): ?>
                <p><strong>Due Date:</strong> <?= formatDate($mission['due_date']) ?></p>
            <?php endif; ?>
            <?php if ($mission['completed_at']): ?>
                <p><strong>Completed:</strong> <?= formatDate($mission['completed_at']) ?></p>
            <?php endif; ?>
        </div>
        
        <?php if ($mission['completion_notes']): ?>
            <div class="form-section">
                <h3>Completion Notes</h3>
                <p><?= nl2br(clean($mission['completion_notes'])) ?></p>
            </div>
        <?php endif; ?>
        
        <?php if ($is_ctd || $is_major_plus): ?>
            <div class="form-section">
                <h3>Update Mission</h3>
                
                <?php if ($mission['status'] === 'pending'): ?>
                    <form method="POST" style="display: inline-block; margin-right: 10px;">
                        <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                        <input type="hidden" name="action" value="update">
                        <input type="hidden" name="status" value="in_progress">
                        <button type="submit" class="btn btn-primary no-loading">Start Mission</button>
                    </form>
                <?php endif; ?>
                
                <?php if ($mission['status'] === 'in_progress'): ?>
                    <form method="POST" style="display: inline-block;">
                        <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                        <input type="hidden" name="action" value="update">
                        <input type="hidden" name="status" value="completed">
                        <textarea name="completion_notes" placeholder="Completion notes..." required style="display: block; margin-bottom: 10px; width: 100%;"></textarea>
                        <button type="submit" class="btn btn-primary no-loading">Complete Mission</button>
                    </form>
                <?php endif; ?>
                
                <?php if (in_array($mission['status'], ['pending', 'in_progress']) && $is_major_plus): ?>
                    <form method="POST" style="display: inline-block; margin-left: 10px;">
                        <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                        <input type="hidden" name="action" value="update">
                        <input type="hidden" name="status" value="cancelled">
                        <button type="submit" class="btn btn-secondary no-loading" onclick="return confirm('Cancel this mission?')">Cancel Mission</button>
                    </form>
                <?php endif; ?>
            </div>
        <?php endif; ?>
        
        <a href="?channel=<?= $mission['channel'] ?>" class="btn btn-secondary">← Back to <?= $mission['channel'] ?></a>
    </div>

<?php else: // List view ?>
    <?php
    $missions = $conn->query("
        SELECT m.*, u.ic_name as assigned_by_name
        FROM ctd_missions m
        JOIN users u ON m.assigned_by = u.id
        WHERE m.channel = '$channel'
        ORDER BY 
            FIELD(m.status, 'pending', 'in_progress', 'completed', 'cancelled'),
            FIELD(m.priority, 'critical', 'high', 'medium', 'low'),
            m.created_at DESC
    ")->fetch_all(MYSQLI_ASSOC);
    ?>
    
    <div class="page-header">
        <h2><?= $channel === 'SRU' ? '🚨 Special Response Unit' : '⚖️ Bail Enforcer Agent' ?> Missions</h2>
    </div>
    
    <?php if (empty($missions)): ?>
        <div class="info-box">
            <p>No missions assigned yet.</p>
        </div>
    <?php else: ?>
        <table class="data-table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Title</th>
                    <th>Priority</th>
                    <th>Assigned To</th>
                    <th>Assigned By</th>
                    <th>Due Date</th>
                    <th>Status</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($missions as $mission): ?>
                    <tr>
                        <td>#<?= $mission['id'] ?></td>
                        <td><strong><?= clean($mission['title']) ?></strong></td>
                        <td>
                            <span class="badge badge-<?= $mission['priority'] === 'critical' ? 'danger' : ($mission['priority'] === 'high' ? 'warning' : 'secondary') ?>">
                                <?= ucfirst($mission['priority']) ?>
                            </span>
                        </td>
                        <td><?= clean($mission['assigned_to']) ?: '-' ?></td>
                        <td><?= clean($mission['assigned_by_name']) ?></td>
                        <td><?= $mission['due_date'] ? formatDate($mission['due_date']) : '-' ?></td>
                        <td>
                            <span class="badge badge-<?= $mission['status'] ?>">
                                <?= ucfirst(str_replace('_', ' ', $mission['status'])) ?>
                            </span>
                        </td>
                        <td>
                            <a href="?action=view&id=<?= $mission['id'] ?>&channel=<?= $channel ?>" class="btn btn-sm">View</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
<?php endif; ?>

<?php include __DIR__ . '/../assets/includes/footer.php'; ?>
