<?php
require_once __DIR__ . '/../config/auth.php';

// Full admin access only for Administrator (level 15)
$is_administrator = ($_SESSION['rank_level'] ?? 0) >= 15;

// Partial admin access for LT-Warden (level 7-14)
$has_partial_admin = ($_SESSION['rank_level'] ?? 0) >= 7 && ($_SESSION['rank_level'] ?? 0) <= 14;

// Check access
if (!$is_administrator && !$has_partial_admin) {
    die('Access denied. Administrator or Command Staff rank required.');
}

$action = $_GET['action'] ?? 'dashboard';
$page = max(1, (int)($_GET['page'] ?? 1));
$per_page = 20;

// Handle user actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($is_administrator || $has_partial_admin)) {
    $token = $_POST['csrf_token'] ?? '';
    if (!verifyToken($token)) die('Invalid token');
    
    $user_action = $_POST['user_action'] ?? '';
    $user_id = (int)($_POST['user_id'] ?? 0);
    
    if ($user_action === 'approve' && $user_id > 0) {
        $stmt = $conn->prepare("UPDATE users SET status = 'active' WHERE id = ?");
        $stmt->bind_param('i', $user_id);
        
        if ($stmt->execute()) {
            // Get user info for notification
            $user_info = $conn->query("SELECT username, ic_name FROM users WHERE id = $user_id")->fetch_assoc();
            
            logActivity('admin_action', "Approved user registration: " . $user_info['username'] . " (" . $user_info['ic_name'] . ")");
            
            $_SESSION['flash_message'] = 'User approved successfully';
            $_SESSION['flash_type'] = 'success';
            redirect('admin?action=pending');
            exit();
        }
    }
    
    if ($user_action === 'reject' && $user_id > 0) {
        // Get user info before deleting
        $user_info = $conn->query("SELECT username, ic_name FROM users WHERE id = $user_id")->fetch_assoc();
        
        // Delete the pending user
        if ($conn->query("DELETE FROM users WHERE id = $user_id")) {
            logActivity('admin_action', "Rejected user registration: " . $user_info['username'] . " (" . $user_info['ic_name'] . ")");
            
            $_SESSION['flash_message'] = 'User rejected and deleted';
            $_SESSION['flash_type'] = 'danger';
            redirect('admin?action=pending');
            exit();
        }
    }
    
    if ($user_action === 'update_rank' && $user_id > 0) {
        $new_rank = (int)$_POST['rank_id'];
        $new_ic_name = sanitizeInput($_POST['ic_name'] ?? '');
        
        // Security check: Can current user manage target user?
        if (!canManageUser($user_id)) {
            die('Access denied. You cannot manage users with equal or higher rank.');
        }
        
        // Security check: Can current user assign this rank?
        if (!canAssignRank($new_rank)) {
            die('Access denied. You cannot assign ranks equal to or higher than your own.');
        }
        
        // Update rank, ic_name, and division (if provided and user is Command+)
        if (isset($_POST['division_id']) && $_SESSION['rank_level'] >= 7) {
            $division_id = (int)$_POST['division_id'];
            $stmt = $conn->prepare("UPDATE users SET rank_id = ?, ic_name = ?, division_id = ? WHERE id = ?");
            $stmt->bind_param('isii', $new_rank, $new_ic_name, $division_id, $user_id);
        } else {
            $stmt = $conn->prepare("UPDATE users SET rank_id = ?, ic_name = ? WHERE id = ?");
            $stmt->bind_param('isi', $new_rank, $new_ic_name, $user_id);
        }
        
        $stmt->execute();
        logActivity('admin_action', "Updated user ID $user_id: Rank to $new_rank, IC Name to '$new_ic_name'");
        redirect('admin?action=users');
    }
    
    if ($user_action === 'update_status' && $user_id > 0) {
        // Security check: Can current user manage target user?
        if (!canManageUser($user_id)) {
            die('Access denied. You cannot manage users with equal or higher rank.');
        }
        
        $new_status = $_POST['status'];
        $stmt = $conn->prepare("UPDATE users SET status = ? WHERE id = ?");
        $stmt->bind_param('si', $new_status, $user_id);
        $stmt->execute();
        logActivity('admin_action', "Updated user status: User ID $user_id to $new_status");
        redirect('admin?action=users');
    }
    
    if ($user_action === 'delete' && $user_id > 0 && $user_id != $_SESSION['user_id']) {
        // Security check: Can current user manage target user?
        if (!canManageUser($user_id)) {
            die('Access denied. You cannot delete users with equal or higher rank.');
        }
        
        $conn->query("DELETE FROM users WHERE id = $user_id");
        logActivity('admin_action', "Deleted user: User ID $user_id");
        redirect('admin?action=users');
    }
}

// Get statistics for admin dashboard
$stats = [];
$stats['total_users'] = $conn->query("SELECT COUNT(*) as count FROM users")->fetch_assoc()['count'];
$stats['active_users'] = $conn->query("SELECT COUNT(*) as count FROM users WHERE status = 'active'")->fetch_assoc()['count'];
$stats['total_threads'] = $conn->query("SELECT COUNT(*) as count FROM threads")->fetch_assoc()['count'];
$stats['total_posts'] = $conn->query("SELECT COUNT(*) as count FROM posts")->fetch_assoc()['count'];
$stats['pending_escorts'] = $conn->query("SELECT COUNT(*) as count FROM escorting_reports WHERE status IN ('draft', 'submitted')")->fetch_assoc()['count'];
$stats['pending_bolingbroke'] = $conn->query("SELECT COUNT(*) as count FROM bolingbroke_activities WHERE status IN ('draft', 'submitted')")->fetch_assoc()['count'];
$stats['pending_paroles'] = $conn->query("SELECT COUNT(*) as count FROM parole_forms WHERE status IN ('draft', 'submitted')")->fetch_assoc()['count'];
$stats['today_activities'] = $conn->query("SELECT COUNT(*) as count FROM activity_logs WHERE DATE(created_at) = CURDATE()")->fetch_assoc()['count'];

$page_title = 'Admin Panel';
include __DIR__ . '/../assets/includes/header.php';
?>

<div class="admin-panel">
    <div class="admin-tabs">
        <a href="?action=dashboard" class="tab <?= $action === 'dashboard' ? 'active' : '' ?>">Dashboard</a>
        <?php if ($is_administrator || $has_partial_admin): ?>
            <?php 
            $pending_count = $conn->query("SELECT COUNT(*) as count FROM users WHERE status = 'pending'")->fetch_assoc()['count'];
            ?>
            <a href="?action=pending" class="tab <?= $action === 'pending' ? 'active' : '' ?>">
                Pending Registrations
                <?php if ($pending_count > 0): ?>
                    <span style="background: #e74c3c; color: white; padding: 2px 8px; border-radius: 12px; font-size: 11px; margin-left: 5px;"><?= $pending_count ?></span>
                <?php endif; ?>
            </a>
            <a href="?action=users" class="tab <?= $action === 'users' ? 'active' : '' ?>">Users</a>
        <?php endif; ?>
        <?php if ($is_administrator): ?>
            <a href="?action=activity" class="tab <?= $action === 'activity' ? 'active' : '' ?>">Activity Logs</a>
        <?php endif; ?>
        <?php if ($is_administrator || $has_partial_admin): ?>
            <a href="?action=forms" class="tab <?= $action === 'forms' ? 'active' : '' ?>">Pending Forms</a>
        <?php endif; ?>
        <?php if ($is_administrator): ?>
            <a href="?action=settings" class="tab <?= $action === 'settings' ? 'active' : '' ?>">Settings</a>
        <?php endif; ?>
    </div>
    
    <?php if ($action === 'dashboard'): ?>
        <div class="page-header">
            <h1>🛡️ Admin Dashboard</h1>
        </div>
        
        <div class="dashboard-stats">
            <div class="stat-card">
                <div class="stat-icon">👥</div>
                <div class="stat-info">
                    <h3><?= $stats['total_users'] ?></h3>
                    <p>Total Users</p>
                    <small><?= $stats['active_users'] ?> active</small>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">📝</div>
                <div class="stat-info">
                    <h3><?= $stats['total_threads'] ?></h3>
                    <p>Total Threads</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">💬</div>
                <div class="stat-info">
                    <h3><?= $stats['total_posts'] ?></h3>
                    <p>Total Posts</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">📋</div>
                <div class="stat-info">
                    <h3><?= $stats['pending_escorts'] + $stats['pending_bolingbroke'] + $stats['pending_paroles'] ?></h3>
                    <p>Pending Forms</p>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">📊</div>
                <div class="stat-info">
                    <h3><?= $stats['today_activities'] ?></h3>
                    <p>Today's Activities</p>
                </div>
            </div>
        </div>
        
        <div class="dashboard-grid">
            <div class="dashboard-section">
                <div class="section-header">
                    <h2>Recent Activity</h2>
                </div>
                
                <?php
                $recent_activity = $conn->query("
                    SELECT a.*, u.ic_name, u.badge_number 
                    FROM activity_logs a
                    LEFT JOIN users u ON a.user_id = u.id
                    ORDER BY a.created_at DESC
                    LIMIT 50
                ")->fetch_all(MYSQLI_ASSOC);
                ?>
                
                <div class="activity-list">
                    <?php foreach ($recent_activity as $activity): ?>
                        <div class="activity-item">
                            <div class="activity-icon">
                                <?php
                                $icon = '•';
                                switch($activity['action']) {
                                    case 'login': $icon = '🔐'; break;
                                    case 'logout': $icon = '🚪'; break;
                                    case 'thread_create': $icon = '📝'; break;
                                    case 'post_create': $icon = '💬'; break;
                                    case 'form_submit': $icon = '📋'; break;
                                    case 'admin_action': $icon = '🛡️'; break;
                                }
                                echo $icon;
                                ?>
                            </div>
                            <div class="activity-content">
                                <strong><?= clean($activity['ic_name'] ?? 'System') ?></strong>
                                <span><?= clean($activity['description'] ?: $activity['action']) ?></span>
                                <small>IP: <?= clean($activity['ip_address']) ?></small>
                            </div>
                            <div class="activity-time">
                                <?= timeAgo($activity['created_at']) ?>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        
    <?php elseif ($action === 'pending' && ($is_administrator || $has_partial_admin)): ?>
        <div class="page-header">
            <h1>⏳ Pending Registrations</h1>
            <p>Review and approve new user registrations</p>
        </div>

        <?php
        $pending_users = $conn->query("
            SELECT u.*, r.name as rank_name, r.display_name as rank_display, r.level as rank_level
            FROM users u
            LEFT JOIN ranks r ON u.rank_id = r.id
            WHERE u.status = 'pending'
            ORDER BY u.created_at DESC
        ")->fetch_all(MYSQLI_ASSOC);
        ?>

        <?php if (empty($pending_users)): ?>
            <div class="info-box">
                <p>✅ No pending registrations at this time.</p>
            </div>
        <?php else: ?>
            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>Registered</th>
                            <th>Username</th>
                            <th>IC Name</th>
                            <th>Badge Number</th>
                            <th>Rank</th>
                            <th>Email</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($pending_users as $user): ?>
                            <tr>
                                <td><?= timeAgo($user['created_at']) ?></td>
                                <td><strong><?= clean($user['username']) ?></strong></td>
                                <td><?= clean($user['ic_name']) ?></td>
                                <td><?= clean($user['badge_number']) ?></td>
                                <td><?= $user['rank_name'] ? clean($user['rank_display'] ?: $user['rank_name']) : '<em style="color: #999;">Not Assigned</em>' ?></td>
                                <td><?= clean($user['email']) ?></td>
                                <td>
                                    <form method="POST" style="display: inline-block;">
                                        <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                                        <input type="hidden" name="user_action" value="approve">
                                        <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                        <button type="submit" class="btn btn-sm no-loading" style="background: #27ae60;" 
                                                onclick="return confirm('Approve this user?');">✓ Approve</button>
                                    </form>
                                    <form method="POST" style="display: inline-block; margin-left: 5px;">
                                        <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                                        <input type="hidden" name="user_action" value="reject">
                                        <input type="hidden" name="user_id" value="<?= $user['id'] ?>">
                                        <button type="submit" class="btn btn-sm no-loading" style="background: #e74c3c;" 
                                                onclick="return confirm('Reject and delete this user?');">✗ Reject</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        <?php endif; ?>

    <?php elseif ($action === 'users' && ($is_administrator || $has_partial_admin)): ?>
        <div class="page-header">
            <h1>👥 User Management</h1>
        </div>
        
        <?php
        $result = $conn->query("SELECT COUNT(*) as total FROM users");
        $total = $result->fetch_assoc()['total'];
        $pagination = paginate($total, $per_page, $page);
        
        $users = $conn->query("
            SELECT u.*, r.display_name as rank_name, r.level as rank_level, d.name as division_name, d.code as division_code
            FROM users u
            JOIN ranks r ON u.rank_id = r.id
            LEFT JOIN divisions d ON u.division_id = d.id
            ORDER BY u.id DESC
            LIMIT {$pagination['offset']}, $per_page
        ")->fetch_all(MYSQLI_ASSOC);
        
        $ranks = $conn->query("SELECT * FROM ranks ORDER BY level")->fetch_all(MYSQLI_ASSOC);
        ?>
        
        <table class="data-table">
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Username</th>
                    <th>IC Name</th>
                    <th>Badge</th>
                    <th>Rank</th>
                    <th>Division</th>
                    <th>Status</th>
                    <th>Last Login</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($users as $user): ?>
                    <tr>
                        <td><?= $user['id'] ?></td>
                        <td><strong><?= clean($user['username']) ?></strong></td>
                        <td><?= clean($user['ic_name']) ?></td>
                        <td><?= clean($user['badge_number']) ?></td>
                        <td>
                            <span class="rank-badge" style="background: <?= getRankColor($user['rank_level']) ?>">
                                <?= clean($user['rank_name']) ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge badge-secondary">
                                <?= clean($user['division_code'] ?? 'OPS') ?>
                            </span>
                        </td>
                        <td>
                            <span class="badge badge-<?= $user['status'] ?>">
                                <?= ucfirst($user['status']) ?>
                            </span>
                        </td>
                        <td><?= $user['last_login'] ? timeAgo($user['last_login']) : 'Never' ?></td>
                        <td>
                            <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                <?php 
                                // Check if current user can manage this user
                                $current_level = $_SESSION['rank_level'] ?? 0;
                                $target_level = $user['rank_level'];
                                // Admin can manage all, Command+ can manage same level or lower
                                $can_manage = ($current_level >= 15) || ($current_level >= 7 && $current_level >= $target_level);
                                ?>
                                <?php if ($can_manage): ?>
                                    <button class="btn btn-sm edit-user-btn" 
                                            data-id="<?= $user['id'] ?>" 
                                            data-username="<?= htmlspecialchars($user['username']) ?>" 
                                            data-icname="<?= htmlspecialchars($user['ic_name']) ?>" 
                                            data-rankid="<?= $user['rank_id'] ?>" 
                                            data-status="<?= $user['status'] ?>" 
                                            data-divisionid="<?= $user['division_id'] ?? 1 ?>">Edit</button>
                                <?php else: ?>
                                    <span class="badge badge-secondary" title="Cannot manage higher or equal rank">🔒 Protected</span>
                                <?php endif; ?>
                            <?php else: ?>
                                <span class="badge badge-info">You</span>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <?php if ($pagination['total_pages'] > 1): ?>
            <div class="pagination">
                <?php for ($i = 1; $i <= $pagination['total_pages']; $i++): ?>
                    <a href="?action=users&page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
                <?php endfor; ?>
            </div>
        <?php endif; ?>
        
        <!-- Edit User Modal -->
        <div id="editUserModal" class="modal" style="display:none;">
            <div class="modal-content">
                <h2>Edit User</h2>
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                    <input type="hidden" name="user_action" value="update_rank">
                    <input type="hidden" name="user_id" id="edit_user_id">
                    
                    <div class="form-group">
                        <label>Username</label>
                        <input type="text" id="edit_username" disabled>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_ic_name">IC Name *</label>
                        <input type="text" name="ic_name" id="edit_ic_name" required placeholder="In-Character Name">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_rank_id">Rank</label>
                        <select name="rank_id" id="edit_rank_id" required>
                            <?php foreach ($ranks as $rank): ?>
                                <?php 
                                // Only show ranks that current user can assign (lower level than current user)
                                $can_assign = ($_SESSION['rank_level'] >= 13) || ($_SESSION['rank_level'] > $rank['level']);
                                ?>
                                <?php if ($can_assign): ?>
                                    <option value="<?= $rank['id'] ?>"><?= clean($rank['display_name']) ?> (Level <?= $rank['level'] ?>)</option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_status">Status</label>
                        <select name="status" id="edit_status" required>
                            <option value="active">Active</option>
                            <option value="inactive">Inactive</option>
                            <option value="suspended">Suspended</option>
                        </select>
                    </div>
                    
                    <?php if ($_SESSION['rank_level'] >= 7): // Only Command and High Command can change division ?>
                        <div class="form-group">
                            <label for="edit_division_id">Division</label>
                            <select name="division_id" id="edit_division_id" required>
                                <?php 
                                $divisions = $conn->query("SELECT * FROM divisions ORDER BY id")->fetch_all(MYSQLI_ASSOC);
                                foreach ($divisions as $div): 
                                ?>
                                    <option value="<?= $div['id'] ?>">
                                        <?= clean($div['name']) ?> (<?= clean($div['code']) ?>)
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    <?php endif; ?>
                    
                    <button type="submit" class="btn btn-primary no-loading">Update User</button>
                    <button type="button" onclick="document.getElementById('editUserModal').style.display='none'" class="btn btn-secondary">Cancel</button>
                </form>
                
                <hr style="margin: 20px 0;">
                
                <form method="POST" onsubmit="return confirm('Update user status?')">
                    <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                    <input type="hidden" name="user_action" value="update_status">
                    <input type="hidden" name="user_id" id="edit_user_id2">
                    <input type="hidden" name="status" id="edit_status2">
                    <button type="submit" class="btn btn-sm no-loading">Update Status</button>
                </form>
                
                <hr style="margin: 20px 0;">
                
                <form method="POST" onsubmit="return confirm('Are you sure you want to delete this user? This action cannot be undone!')">
                    <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                    <input type="hidden" name="user_action" value="delete">
                    <input type="hidden" name="user_id" id="edit_user_id3">
                    <button type="submit" class="btn btn-danger btn-sm no-loading">Delete User</button>
                </form>
            </div>
        </div>
        
        <script>
        // Edit user function
        function editUser(id, username, icName, rankId, status, divisionId) {
            document.getElementById('edit_user_id').value = id;
            document.getElementById('edit_user_id2').value = id;
            document.getElementById('edit_user_id3').value = id;
            document.getElementById('edit_username').value = username;
            document.getElementById('edit_ic_name').value = icName;
            document.getElementById('edit_rank_id').value = rankId;
            document.getElementById('edit_status').value = status;
            document.getElementById('edit_status2').value = status;
            if (document.getElementById('edit_division_id')) {
                document.getElementById('edit_division_id').value = divisionId || 1;
            }
            document.getElementById('editUserModal').style.display = 'block';
        }
        
        // Event listener for edit buttons using data attributes
        document.addEventListener('DOMContentLoaded', function() {
            document.querySelectorAll('.edit-user-btn').forEach(function(btn) {
                btn.addEventListener('click', function() {
                    const id = this.dataset.id;
                    const username = this.dataset.username;
                    const icName = this.dataset.icname;
                    const rankId = this.dataset.rankid;
                    const status = this.dataset.status;
                    const divisionId = this.dataset.divisionid;
                    editUser(id, username, icName, rankId, status, divisionId);
                });
            });
        });
        </script>
        
    <?php elseif ($action === 'activity'): ?>
        <?php
        // Only Administrator (level 15) can view activity logs
        if (!$is_administrator) {
            die('<div class="container"><div class="alert alert-error">Access denied. Activity logs are only accessible to Administrators (Level 15).</div></div>');
        }
        
        // Handle delete logs by date
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_logs'])) {
            validateToken();
            
            $delete_date = $_POST['delete_date'] ?? '';
            
            if (!empty($delete_date)) {
                $stmt = $conn->prepare("DELETE FROM activity_logs WHERE DATE(created_at) = ?");
                $stmt->bind_param('s', $delete_date);
                $stmt->execute();
                $deleted_count = $stmt->affected_rows;
                
                logActivity('admin_action', "Deleted $deleted_count activity logs from date: $delete_date");
                
                // Store success message in session and redirect
                $_SESSION['admin_message'] = "Successfully deleted $deleted_count activity log(s) from $delete_date";
                redirect('admin?action=activity');
            }
        }
        
        // Handle delete all logs
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_all_logs'])) {
            validateToken();
            
            $total_logs = $conn->query("SELECT COUNT(*) as count FROM activity_logs")->fetch_assoc()['count'];
            $conn->query("TRUNCATE TABLE activity_logs");
            
            logActivity('admin_action', "Deleted ALL activity logs ($total_logs records)");
            
            // Store success message in session and redirect
            $_SESSION['admin_message'] = "Successfully deleted ALL $total_logs activity log(s)";
            $_SESSION['admin_message_type'] = 'warning';
            redirect('admin?action=activity');
        }
        ?>
        
        <div class="page-header">
            <h1>📊 Activity Logs</h1>
        </div>
        
        <?php
        // Display session message if exists
        if (isset($_SESSION['admin_message'])) {
            $msg_type = $_SESSION['admin_message_type'] ?? 'info';
            echo '<div class="alert alert-' . $msg_type . '" style="margin-bottom: 20px;">';
            echo clean($_SESSION['admin_message']);
            echo '</div>';
            unset($_SESSION['admin_message']);
            unset($_SESSION['admin_message_type']);
        }
        ?>
        
        <!-- Delete Logs Controls -->
        <div style="background: var(--card-bg); padding: 20px; border-radius: 8px; margin-bottom: 20px; border: 2px solid var(--danger);">
            <h3 style="color: var(--danger); margin-bottom: 15px;">🗑️ Delete Activity Logs</h3>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px;">
                <!-- Delete by Date -->
                <form method="POST" onsubmit="return confirm('Are you sure you want to delete all activity logs from this date? This action cannot be undone!');">
                    <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                    <div class="form-group">
                        <label for="delete_date">Delete Logs by Date:</label>
                        <input type="date" id="delete_date" name="delete_date" required style="margin-bottom: 10px;">
                        <button type="submit" name="delete_logs" class="btn btn-danger btn-sm no-loading">🗑️ Delete Logs from Selected Date</button>
                    </div>
                </form>
                
                <!-- Delete All -->
                <form method="POST" onsubmit="return confirm('⚠️ WARNING: This will DELETE ALL activity logs permanently! Are you absolutely sure?');">
                    <input type="hidden" name="csrf_token" value="<?= generateToken() ?>">
                    <div class="form-group">
                        <label style="color: var(--danger); font-weight: bold;">⚠️ Danger Zone:</label>
                        <p style="font-size: 13px; color: var(--text-light); margin: 8px 0;">This will permanently delete ALL activity logs from the database.</p>
                        <button type="submit" name="delete_all_logs" class="btn btn-danger btn-sm no-loading">🗑️ Delete ALL Activity Logs</button>
                    </div>
                </form>
            </div>
        </div>
        
        <?php
        $result = $conn->query("SELECT COUNT(*) as total FROM activity_logs");
        $total = $result->fetch_assoc()['total'];
        $pagination = paginate($total, $per_page, $page);
        
        $activities = $conn->query("
            SELECT a.*, u.ic_name, u.badge_number 
            FROM activity_logs a
            LEFT JOIN users u ON a.user_id = u.id
            ORDER BY a.created_at DESC
            LIMIT {$pagination['offset']}, $per_page
        ")->fetch_all(MYSQLI_ASSOC);
        ?>
        
        <table class="data-table">
            <thead>
                <tr>
                    <th>Time</th>
                    <th>User</th>
                    <th>Action</th>
                    <th>Description</th>
                    <th>IP Address</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($activities as $activity): ?>
                    <tr>
                        <td><?= formatDate($activity['created_at']) ?></td>
                        <td>
                            <?php if ($activity['ic_name']): ?>
                                <?= clean($activity['ic_name']) ?><br>
                                <small>#<?= clean($activity['badge_number']) ?></small>
                            <?php else: ?>
                                <em>System</em>
                            <?php endif; ?>
                        </td>
                        <td><code><?= clean($activity['action']) ?></code></td>
                        <td><?= clean($activity['description']) ?></td>
                        <td><?= clean($activity['ip_address']) ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <?php if ($pagination['total_pages'] > 1): ?>
            <div class="pagination">
                <?php for ($i = 1; $i <= $pagination['total_pages']; $i++): ?>
                    <a href="?action=activity&page=<?= $i ?>" class="<?= $i === $page ? 'active' : '' ?>"><?= $i ?></a>
                <?php endfor; ?>
            </div>
        <?php endif; ?>
        
    <?php elseif ($action === 'forms'): ?>
        <div class="page-header">
            <h1>📋 Pending Forms Review</h1>
        </div>
        
        <div class="admin-forms">
            <div class="form-section">
                <h2>Pending Escorting Reports (<?= $stats['pending_escorts'] ?>)</h2>
                <?php
                $pending_escorts = $conn->query("
                    SELECT e.*, u.ic_name, u.badge_number 
                    FROM escorting_reports e
                    JOIN users u ON e.user_id = u.id
                    WHERE e.status IN ('draft', 'submitted')
                    ORDER BY e.created_at DESC
                    LIMIT 10
                ")->fetch_all(MYSQLI_ASSOC);
                ?>
                
                <?php if (count($pending_escorts) > 0): ?>
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Officer</th>
                                <th>IC Escorting</th>
                                <th>From → To</th>
                                <th>Time</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($pending_escorts as $escort): ?>
                                <tr>
                                    <td><?= clean($escort['ic_name']) ?></td>
                                    <td><?= clean($escort['ic_escorting']) ?></td>
                                    <td><?= clean($escort['from_location']) ?> → <?= clean($escort['to_location']) ?></td>
                                    <td><?= formatDate($escort['escort_time']) ?></td>
                                    <td>
                                        <a href="/doc/forms/escorting_report.php?action=view&id=<?= $escort['id'] ?>" class="btn btn-sm">View</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p class="no-data">No pending escorting reports</p>
                <?php endif; ?>
            </div>
            
            <div class="form-section">
                <h2>Pending BolingBroke Activities (<?= $stats['pending_bolingbroke'] ?>)</h2>
                <?php
                $pending_bolingbroke = $conn->query("
                    SELECT b.*, u.ic_name, u.badge_number 
                    FROM bolingbroke_activities b
                    JOIN users u ON b.user_id = u.id
                    WHERE b.status IN ('draft', 'submitted')
                    ORDER BY b.created_at DESC
                    LIMIT 10
                ")->fetch_all(MYSQLI_ASSOC);
                ?>
                
                <?php if (count($pending_bolingbroke) > 0): ?>
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Officer</th>
                                <th>Shift Leader</th>
                                <th>Date/Time</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($pending_bolingbroke as $activity): ?>
                                <tr>
                                    <td><strong>#<?= $activity['id'] ?></strong></td>
                                    <td><?= clean($activity['ic_name']) ?></td>
                                    <td><?= clean($activity['shift_leader']) ?></td>
                                    <td><?= formatDate($activity['activity_datetime']) ?></td>
                                    <td>
                                        <a href="/doc/forms/bolingbroke_activity.php?action=view&id=<?= $activity['id'] ?>" class="btn btn-sm">View</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p class="no-data">No pending bolingbroke activities</p>
                <?php endif; ?>
            </div>
            
            <div class="form-section">
                <h2>Pending Parole Forms (<?= $stats['pending_paroles'] ?>)</h2>
                <?php
                $pending_paroles = $conn->query("
                    SELECT p.*, u.ic_name, u.badge_number 
                    FROM parole_forms p
                    JOIN users u ON p.user_id = u.id
                    WHERE p.status IN ('draft', 'submitted')
                    ORDER BY p.created_at DESC
                    LIMIT 10
                ")->fetch_all(MYSQLI_ASSOC);
                ?>
                
                <?php if (count($pending_paroles) > 0): ?>
                    <table class="data-table">
                        <thead>
                            <tr>
                                <th>Officer</th>
                                <th>Inmate</th>
                                <th>Charge</th>
                                <th>End of Parole</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($pending_paroles as $parole): ?>
                                <tr>
                                    <td><?= clean($parole['ic_name']) ?></td>
                                    <td><?= clean($parole['inmate_name']) ?></td>
                                    <td><?= clean($parole['charge']) ?></td>
                                    <td><?= formatDate($parole['end_of_parole'], 'M d, Y') ?></td>
                                    <td>
                                        <a href="/doc/forms/parole_form.php?action=view&id=<?= $parole['id'] ?>" class="btn btn-sm">View</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p class="no-data">No pending parole forms</p>
                <?php endif; ?>
            </div>
        </div>
        
    <?php elseif ($action === 'settings' && $is_administrator): ?>
        <div class="page-header">
            <h1>⚙️ System Settings</h1>
        </div>
        
        <div class="settings-panel">
            <div class="setting-card">
                <h3>System Information</h3>
                <table class="info-table">
                    <tr>
                        <td><strong>PHP Version:</strong></td>
                        <td><?= phpversion() ?></td>
                    </tr>
                    <tr>
                        <td><strong>Database:</strong></td>
                        <td><?= $_ENV['DB_NAME'] ?? 'doc_ic_forum' ?></td>
                    </tr>
                    <tr>
                        <td><strong>Session Lifetime:</strong></td>
                        <td><?= $_ENV['SESSION_LIFETIME'] ?? '10800' ?> seconds (<?= round(($_ENV['SESSION_LIFETIME'] ?? 10800) / 3600, 1) ?> hours)</td>
                    </tr>
                    <tr>
                        <td><strong>Site Name:</strong></td>
                        <td><?= $_ENV['SITE_NAME'] ?? 'DOC IC Forum' ?></td>
                    </tr>
                </table>
            </div>
            
            <div class="setting-card">
                <h3>Database Statistics</h3>
                <table class="info-table">
                    <tr>
                        <td><strong>Total Users:</strong></td>
                        <td><?= $stats['total_users'] ?></td>
                    </tr>
                    <tr>
                        <td><strong>Total Threads:</strong></td>
                        <td><?= $stats['total_threads'] ?></td>
                    </tr>
                    <tr>
                        <td><strong>Total Posts:</strong></td>
                        <td><?= $stats['total_posts'] ?></td>
                    </tr>
                    <tr>
                        <td><strong>Activity Logs:</strong></td>
                        <td><?= $conn->query("SELECT COUNT(*) as count FROM activity_logs")->fetch_assoc()['count'] ?></td>
                    </tr>
                </table>
            </div>
        </div>
    <?php endif; ?>
</div>

<style>
.admin-panel {
    min-height: 60vh;
}

.admin-tabs {
    display: flex;
    gap: 10px;
    margin-bottom: 30px;
    border-bottom: 2px solid var(--border);
}

.admin-tabs .tab {
    padding: 12px 24px;
    text-decoration: none;
    color: var(--text);
    border-bottom: 3px solid transparent;
    transition: all 0.3s;
    font-weight: 600;
}

.admin-tabs .tab:hover {
    background: var(--light);
    color: var(--primary);
}

.admin-tabs .tab.active {
    color: var(--primary);
    border-bottom-color: var(--primary);
}

.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
}

.modal-content {
    background: white;
    padding: 30px;
    border-radius: 12px;
    max-width: 500px;
    width: 90%;
    max-height: 80vh;
    overflow-y: auto;
}

.admin-forms .form-section {
    margin-bottom: 40px;
}

.admin-forms .form-section h2 {
    color: var(--primary);
    margin-bottom: 15px;
}

.settings-panel {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 20px;
}

.setting-card {
    background: white;
    padding: 25px;
    border-radius: 12px;
    box-shadow: 0 2px 15px rgba(0,0,0,0.1);
}

.setting-card h3 {
    color: var(--primary);
    margin-bottom: 15px;
    border-bottom: 2px solid var(--border);
    padding-bottom: 10px;
}

.info-table {
    width: 100%;
}

.info-table tr {
    border-bottom: 1px solid var(--border);
}

.info-table td {
    padding: 10px 5px;
}
</style>

<?php include __DIR__ . '/../assets/includes/footer.php'; ?>
