<?php
require_once __DIR__ . '/session.php';

// Check if user has required rank level
function hasRank($required_level) {
    if (!isset($_SESSION['user_id'])) {
        return false;
    }
    
    $user_level = $_SESSION['rank_level'] ?? 0;
    return $user_level >= $required_level;
}

// Check if user has specific permission
function hasPermission($permission) {
    if (!isset($_SESSION['user_id'])) {
        return false;
    }
    
    // Administrator (level 15) has all permissions
    if (hasRank(15)) {
        return true;
    }
    
    $permissions = $_SESSION['permissions'] ?? [];
    
    // Check for wildcard permission
    if (in_array('*', $permissions)) {
        return true;
    }
    
    // Check exact match
    if (in_array($permission, $permissions)) {
        return true;
    }
    
    // Check wildcard in category (e.g., 'forum.*')
    $parts = explode('.', $permission);
    if (count($parts) > 1) {
        $wildcard = $parts[0] . '.*';
        if (in_array($wildcard, $permissions)) {
            return true;
        }
    }
    
    return false;
}

// Require authentication
function requireAuth() {
    if (!validateSession()) {
        header('Location: /doc/modules/login.php');
        exit;
    }
}

// Require specific rank
function requireRank($level) {
    requireAuth();
    if (!hasRank($level)) {
        die('Access denied. Insufficient rank.');
    }
}

// Require specific permission
function requirePermission($permission) {
    requireAuth();
    if (!hasPermission($permission)) {
        die('Access denied. Missing permission: ' . htmlspecialchars($permission));
    }
}

// Log activity
function logActivity($action, $description = '') {
    global $conn;
    
    if (!isset($_SESSION['user_id'])) return;
    
    $user_id = $_SESSION['user_id'];
    $ip = $_SERVER['REMOTE_ADDR'] ?? '';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    
    $stmt = $conn->prepare(
        "INSERT INTO activity_logs (user_id, action, description, ip_address, user_agent) 
         VALUES (?, ?, ?, ?, ?)"
    );
    $stmt->bind_param('issss', $user_id, $action, $description, $ip, $user_agent);
    $stmt->execute();
}

// Check if current user can manage target user
// Returns true only if current user's rank level is HIGHER than target user
function canManageUser($target_user_id) {
    global $conn;
    
    if (!isset($_SESSION['user_id'])) {
        return false;
    }
    
    // Cannot manage yourself (except Administrator can)
    if ($_SESSION['user_id'] == $target_user_id && $_SESSION['rank_level'] < 13) {
        return false;
    }
    
    // Get target user's rank level
    $stmt = $conn->prepare("
        SELECT r.level 
        FROM users u 
        JOIN ranks r ON u.rank_id = r.id 
        WHERE u.id = ?
    ");
    $stmt->bind_param('i', $target_user_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $target = $result->fetch_assoc();
    
    if (!$target) {
        return false;
    }
    
    $current_level = $_SESSION['rank_level'] ?? 0;
    $target_level = $target['level'];
    
    // Administrator (level 15) can manage anyone
    if ($current_level >= 15) {
        return true;
    }
    
    // Can only manage users with LOWER rank level (strict hierarchy)
    return $current_level > $target_level;
}

// Check if target rank is valid for current user to assign
// Returns true only if target rank level is LOWER than current user
function canAssignRank($target_rank_id) {
    global $conn;
    
    if (!isset($_SESSION['user_id'])) {
        return false;
    }
    
    // Get target rank level
    $stmt = $conn->prepare("SELECT level FROM ranks WHERE id = ?");
    $stmt->bind_param('i', $target_rank_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $rank = $result->fetch_assoc();
    
    if (!$rank) {
        return false;
    }
    
    $current_level = $_SESSION['rank_level'] ?? 0;
    $target_level = $rank['level'];
    
    // Administrator (level 15) can assign any rank
    if ($current_level >= 15) {
        return true;
    }
    
    // Can only assign ranks LOWER than your own level
    return $current_level > $target_level;
}