<?php
// Sanitize output for HTML
function clean($str) {
    return htmlspecialchars($str, ENT_QUOTES | ENT_HTML5, 'UTF-8');
}

// Sanitize input
function sanitizeInput($input) {
    if (is_array($input)) {
        return array_map('sanitizeInput', $input);
    }
    return trim(strip_tags($input));
}

// Format date for display
function formatDate($datetime, $format = 'M d, Y H:i') {
    if (!$datetime) return 'N/A';
    $dt = new DateTime($datetime);
    return $dt->format($format);
}

// Time ago function
function timeAgo($datetime) {
    $time = strtotime($datetime);
    $diff = time() - $time;
    
    if ($diff < 60) return 'just now';
    if ($diff < 3600) return floor($diff / 60) . ' minutes ago';
    if ($diff < 86400) return floor($diff / 3600) . ' hours ago';
    if ($diff < 604800) return floor($diff / 86400) . ' days ago';
    
    return date('M d, Y', $time);
}

// Generate CSRF token
function generateToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

// Verify CSRF token
function verifyToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

// Validate CSRF token from POST request (throws error if invalid)
function validateToken() {
    $token = $_POST['csrf_token'] ?? '';
    if (!verifyToken($token)) {
        die('Invalid CSRF token. Please try again.');
    }
}

// Redirect helper
function redirect($path) {
    header('Location: ' . $path);
    exit;
}

// Get rank badge color
function getRankColor($level) {
    $colors = [
        1 => '#6c757d',  // Trainee - Gray
        2 => '#5a6268',  // Correctional Officer - Dark Gray
        3 => '#007bff',  // Senior CO - Blue
        4 => '#17a2b8',  // Corporal - Cyan
        5 => '#28a745',  // Sergeant - Green
        6 => '#20c997',  // Sergeant Overwatch - Teal
        7 => '#ffc107',  // Lieutenant - Yellow
        8 => '#fd7e14',  // Captain - Orange
        9 => '#e83e8c',  // Commaderi (Commander PSB) - Pink
        10 => '#dc3545', // Commandera (Commander COD) - Red
        11 => '#6f42c1', // majora (Major DOFO) - Purple
        12 => '#8b4789',  // majorb (Major ADMIN) - Violet
        13 => '#5a189a', // Deputy Chief Warden - Dark Purple
        14 => '#343a40', // Warden - Dark
        15 => '#000000'  // Administrator - Black
    ];
    return $colors[$level] ?? '#6c757d';
}

// Pagination helper
function paginate($total, $perPage, $currentPage) {
    $totalPages = ceil($total / $perPage);
    $currentPage = max(1, min($currentPage, $totalPages));
    $offset = ($currentPage - 1) * $perPage;
    
    return [
        'total' => $total,
        'per_page' => $perPage,
        'current_page' => $currentPage,
        'total_pages' => $totalPages,
        'offset' => $offset
    ];
}